/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.pipe.consensus.deletion.persist;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.iotdb.commons.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.consensus.index.ProgressIndex;
import org.apache.iotdb.commons.consensus.index.impl.MinimumProgressIndex;
import org.apache.iotdb.commons.consensus.index.impl.SimpleProgressIndex;
import org.apache.iotdb.commons.utils.TestOnly;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.pipe.consensus.ReplicateProgressDataNodeManager;
import org.apache.iotdb.db.pipe.consensus.deletion.DeletionResource;
import org.apache.iotdb.db.pipe.consensus.deletion.persist.DeletionBuffer;
import org.apache.iotdb.db.utils.MmapUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PageCacheDeletionBuffer
implements DeletionBuffer {
    private static final Logger LOGGER = LoggerFactory.getLogger(PageCacheDeletionBuffer.class);
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final double FSYNC_BUFFER_RATIO = 0.95;
    private static final int QUEUE_CAPACITY = config.getDeletionAheadLogBufferQueueCapacity();
    private static final long MAX_WAIT_CLOSE_TIME_IN_MS = 10000L;
    public static int DAL_BUFFER_SIZE = config.getWalBufferSize() / 3;
    private final BlockingQueue<DeletionResource> deletionResources = new PriorityBlockingQueue<DeletionResource>(QUEUE_CAPACITY, (o1, o2) -> o1.getProgressIndex().equals(o2.getProgressIndex()) ? 0 : (o1.getProgressIndex().isAfter(o2.getProgressIndex()) ? 1 : -1));
    private final String dataRegionId;
    private final String baseDirectory;
    private final ExecutorService persistThread;
    private final Lock buffersLock = new ReentrantLock();
    private final AtomicInteger totalSize = new AtomicInteger(0);
    private final List<DeletionResource> pendingDeletionsInOneTask = new CopyOnWriteArrayList<DeletionResource>();
    private volatile boolean isClosed = false;
    private volatile ByteBuffer serializeBuffer;
    private volatile File logFile;
    private volatile FileOutputStream logStream;
    private volatile FileChannel logChannel;
    private ProgressIndex maxProgressIndexInCurrentFile = MinimumProgressIndex.INSTANCE;

    public PageCacheDeletionBuffer(String dataRegionId, String baseDirectory) {
        this.dataRegionId = dataRegionId;
        this.baseDirectory = baseDirectory;
        this.allocateBuffers();
        this.persistThread = IoTDBThreadPoolFactory.newSingleThreadExecutor((String)(ThreadName.PIPE_CONSENSUS_DELETION_SERIALIZE.getName() + "(group-" + dataRegionId + ")"));
    }

    @Override
    public void start() {
        this.persistThread.submit(new PersistTask());
        try {
            this.logFile = new File(this.baseDirectory, String.format("_%d-%d%s", 0, 0, ".deletion"));
            this.logStream = new FileOutputStream(this.logFile, true);
            this.logChannel = this.logStream.getChannel();
            if (!this.logFile.exists() || this.logFile.length() == 0L) {
                this.logChannel.write(ByteBuffer.wrap("DELETION_V1".getBytes(StandardCharsets.UTF_8)));
            }
            LOGGER.info("Deletion persist-{}: starting to persist, current writing: {}", (Object)this.dataRegionId, (Object)this.logFile);
        }
        catch (IOException e) {
            LOGGER.warn("Deletion persist: Cannot create file {}, please check your file system manually.", (Object)this.logFile, (Object)e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public boolean isAllDeletionFlushed() {
        this.buffersLock.lock();
        try {
            int pos = Optional.ofNullable(this.serializeBuffer).map(Buffer::position).orElse(0);
            boolean bl = this.deletionResources.isEmpty() && pos == 0;
            return bl;
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    private void allocateBuffers() {
        try {
            this.serializeBuffer = ByteBuffer.allocateDirect(DAL_BUFFER_SIZE);
        }
        catch (OutOfMemoryError e) {
            LOGGER.error("Fail to allocate deletionBuffer-group-{}'s buffer because out of memory.", (Object)this.dataRegionId, (Object)e);
            this.close();
            throw e;
        }
    }

    @Override
    public void registerDeletionResource(DeletionResource deletionResource) {
        if (this.isClosed) {
            LOGGER.error("Fail to register DeletionResource into deletionBuffer-{} because this buffer is closed.", (Object)this.dataRegionId);
            return;
        }
        this.deletionResources.add(deletionResource);
    }

    private void appendCurrentBatch() throws IOException {
        this.serializeBuffer.flip();
        this.logChannel.write(this.serializeBuffer);
    }

    private void fsyncCurrentLoggingFile() throws IOException {
        LOGGER.info("Deletion persist-{}: current batch fsync due to timeout", (Object)this.dataRegionId);
        this.logChannel.force(false);
        this.pendingDeletionsInOneTask.forEach(DeletionResource::onPersistSucceed);
    }

    private void closeCurrentLoggingFile() throws IOException {
        LOGGER.info("Deletion persist-{}: current file has been closed", (Object)this.dataRegionId);
        this.logStream.close();
        this.logChannel.close();
        this.pendingDeletionsInOneTask.forEach(DeletionResource::onPersistSucceed);
    }

    private void resetTaskAttribute() {
        this.pendingDeletionsInOneTask.clear();
        this.clearBuffer();
    }

    private void resetFileAttribute() {
        this.totalSize.set(0);
        this.maxProgressIndexInCurrentFile = MinimumProgressIndex.INSTANCE;
    }

    private void rollbackFileAttribute(int currentBatchSize) {
        this.totalSize.addAndGet(-currentBatchSize);
    }

    private void clearBuffer() {
        this.buffersLock.lock();
        try {
            this.serializeBuffer.clear();
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    private void switchLoggingFile() throws IOException {
        try {
            ProgressIndex curProgressIndex = ReplicateProgressDataNodeManager.extractLocalSimpleProgressIndex(this.maxProgressIndexInCurrentFile);
            if (!(curProgressIndex instanceof SimpleProgressIndex)) {
                throw new IOException("Invalid deletion progress index: " + curProgressIndex);
            }
            SimpleProgressIndex progressIndex = (SimpleProgressIndex)curProgressIndex;
            this.logFile = new File(this.baseDirectory, String.format("_%d-%d%s", progressIndex.getRebootTimes(), progressIndex.getMemTableFlushOrderId(), ".deletion"));
            this.logStream = new FileOutputStream(this.logFile, true);
            this.logChannel = this.logStream.getChannel();
            if (!this.logFile.exists() || this.logFile.length() == 0L) {
                this.logChannel.write(ByteBuffer.wrap("DELETION_V1".getBytes(StandardCharsets.UTF_8)));
            }
            LOGGER.info("Deletion persist-{}: switching to a new file, current writing: {}", (Object)this.dataRegionId, (Object)this.logFile);
        }
        finally {
            this.resetFileAttribute();
        }
    }

    @Override
    public void close() {
        this.isClosed = true;
        this.waitUntilFlushAllDeletionsOrTimeOut();
        if (this.persistThread != null) {
            this.persistThread.shutdownNow();
            try {
                if (!this.persistThread.awaitTermination(30L, TimeUnit.SECONDS)) {
                    LOGGER.warn("persistThread did not terminate within {}s", (Object)30);
                }
            }
            catch (InterruptedException e) {
                LOGGER.warn("DAL Thread {} still doesn't exit after 30s", (Object)this.dataRegionId);
                Thread.currentThread().interrupt();
            }
        }
        MmapUtil.clean(this.serializeBuffer);
        this.serializeBuffer = null;
    }

    private void waitUntilFlushAllDeletionsOrTimeOut() {
        long currentTime = System.currentTimeMillis();
        while (!this.isAllDeletionFlushed() && System.currentTimeMillis() - currentTime < 10000L) {
            try {
                Thread.sleep(50L);
            }
            catch (InterruptedException e) {
                LOGGER.error("Interrupted when waiting for all deletions flushed.");
                Thread.currentThread().interrupt();
            }
        }
    }

    @TestOnly
    public static void setDalBufferSize(int dalBufferSize) {
        DAL_BUFFER_SIZE = dalBufferSize;
    }

    private class PersistTask
    implements Runnable {
        private final AtomicInteger currentTaskBatchSize = new AtomicInteger(0);

        private PersistTask() {
        }

        @Override
        public void run() {
            try {
                this.persistDeletion();
            }
            catch (IOException e) {
                LOGGER.warn("Deletion persist: Cannot write to {}, may cause data inconsistency.", (Object)PageCacheDeletionBuffer.this.logFile, (Object)e);
                PageCacheDeletionBuffer.this.pendingDeletionsInOneTask.forEach(deletionResource -> deletionResource.onPersistFailed(e));
                PageCacheDeletionBuffer.this.rollbackFileAttribute(this.currentTaskBatchSize.get());
            }
            finally {
                if (!PageCacheDeletionBuffer.this.isClosed) {
                    PageCacheDeletionBuffer.this.persistThread.submit(new PersistTask());
                }
            }
        }

        private boolean serializeDeletionToBatchBuffer(DeletionResource deletionResource) {
            LOGGER.debug("Deletion persist-{}: serialize deletion resource {}", (Object)PageCacheDeletionBuffer.this.dataRegionId, (Object)deletionResource);
            ByteBuffer buffer = deletionResource.serialize();
            if (buffer.position() > PageCacheDeletionBuffer.this.serializeBuffer.remaining()) {
                return false;
            }
            PageCacheDeletionBuffer.this.serializeBuffer.put(buffer.array());
            PageCacheDeletionBuffer.this.totalSize.addAndGet(buffer.position());
            this.currentTaskBatchSize.addAndGet(buffer.position());
            return true;
        }

        private void persistDeletion() throws IOException {
            try {
                DeletionResource firstDeletionResource = (DeletionResource)PageCacheDeletionBuffer.this.deletionResources.take();
                this.serializeDeletionToBatchBuffer(firstDeletionResource);
                PageCacheDeletionBuffer.this.pendingDeletionsInOneTask.add(firstDeletionResource);
                PageCacheDeletionBuffer.this.maxProgressIndexInCurrentFile = PageCacheDeletionBuffer.this.maxProgressIndexInCurrentFile.updateToMinimumEqualOrIsAfterProgressIndex(firstDeletionResource.getProgressIndex());
            }
            catch (InterruptedException e) {
                LOGGER.warn("Interrupted when waiting for taking DeletionResource from blocking queue to serialize.");
                Thread.currentThread().interrupt();
                return;
            }
            while ((double)PageCacheDeletionBuffer.this.totalSize.get() < (double)DAL_BUFFER_SIZE * 0.95) {
                DeletionResource deletionResource = null;
                try {
                    deletionResource = (DeletionResource)PageCacheDeletionBuffer.this.deletionResources.poll(config.getWalAsyncModeFsyncDelayInMs(), TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    LOGGER.warn("Interrupted when waiting for taking WALEntry from blocking queue to serialize.");
                    Thread.currentThread().interrupt();
                }
                if (deletionResource == null) {
                    PageCacheDeletionBuffer.this.appendCurrentBatch();
                    PageCacheDeletionBuffer.this.fsyncCurrentLoggingFile();
                    PageCacheDeletionBuffer.this.resetTaskAttribute();
                    return;
                }
                if (!this.serializeDeletionToBatchBuffer(deletionResource)) {
                    PageCacheDeletionBuffer.this.deletionResources.add(deletionResource);
                    PageCacheDeletionBuffer.this.appendCurrentBatch();
                    PageCacheDeletionBuffer.this.closeCurrentLoggingFile();
                    PageCacheDeletionBuffer.this.resetTaskAttribute();
                    PageCacheDeletionBuffer.this.switchLoggingFile();
                    return;
                }
                PageCacheDeletionBuffer.this.pendingDeletionsInOneTask.add(deletionResource);
                PageCacheDeletionBuffer.this.maxProgressIndexInCurrentFile = PageCacheDeletionBuffer.this.maxProgressIndexInCurrentFile.updateToMinimumEqualOrIsAfterProgressIndex(deletionResource.getProgressIndex());
            }
            if (PageCacheDeletionBuffer.this.totalSize.get() > 0) {
                PageCacheDeletionBuffer.this.appendCurrentBatch();
                PageCacheDeletionBuffer.this.closeCurrentLoggingFile();
                PageCacheDeletionBuffer.this.resetTaskAttribute();
                PageCacheDeletionBuffer.this.switchLoggingFile();
            }
        }
    }
}

