/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau;

import static java.util.Collections.*;
import static org.apache.juneau.commons.utils.AssertionUtils.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.StringUtils.*;
import static org.apache.juneau.commons.utils.ThrowableUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;

import java.text.*;
import java.util.*;
import java.util.function.*;

import org.apache.juneau.commons.collections.*;
import org.apache.juneau.commons.function.*;
import org.apache.juneau.commons.reflect.*;

/**
 * A one-time-use non-thread-safe object that's meant to be used once and then thrown away.
 *
 * <h5 class='section'>Notes:</h5><ul>
 * 	<li class='warn'>This class is not typically thread safe.
 * </ul>
 *
 */
public abstract class ContextSession {

	/**
	 * Builder class.
	 */
	public static abstract class Builder {
		private Boolean debug;
		private boolean unmodifiable;
		private Context ctx;
		private ResettableSupplier<LinkedHashMap<String,Object>> properties;

		/**
		 * Constructor.
		 *
		 * @param ctx The context creating this session.
		 * 	<br>Cannot be <jk>null</jk>.
		 */
		protected Builder(Context ctx) {
			this.ctx = assertArgNotNull("ctx", ctx);
			this.properties = memr(LinkedHashMap::new);
		}

		/**
		 * Applies a consumer to this builder if it's the specified type.
		 *
		 * @param <T> The expected type.
		 * @param type The expected type.
		 * 	<br>Cannot be <jk>null</jk>.
		 * @param apply	The consumer to apply.
		 * 	<br>Cannot be <jk>null</jk>.
		 * @return This object.
		 */
		public <T> Builder apply(Class<T> type, Consumer<T> apply) {
			if (assertArgNotNull("type", type).isInstance(this))
				assertArgNotNull("apply", apply).accept(type.cast(this));
			return this;
		}

		/**
		 * Build the object.
		 *
		 * @return The built object.
		 */
		public abstract ContextSession build();

		/**
		 * Debug mode.
		 *
		 * <p>
		 * Enables the following additional information during parsing:
		 * <ul>
		 * 	<li> When bean setters throws exceptions, the exception includes the object stack information in order to determine how that method was invoked.
		 * </ul>
		 *
		 * <p>
		 * If not specified, defaults to {@link Context.Builder#debug()}.
		 *
		 * <h5 class='section'>See Also:</h5><ul>
		 * 	<li class='ja'>{@link org.apache.juneau.annotation.BeanConfig#debug()}
		 * 	<li class='jm'>{@link org.apache.juneau.Context.Builder#debug()}
		 *
		 * @param value
		 * 	The new value for this property.
		 * 	<br>If <jk>null</jk>, defaults to {@link Context#isDebug()}.
		 * @return This object.
		 */
		public Builder debug(Boolean value) {
			debug = value;
			return this;
		}

		/**
		 * Session properties.
		 *
		 * <p>
		 * Session properties are generic key-value pairs that can be passed through the session and made
		 * available to any customized serializers/parsers or swaps.
		 *
		 * @param value
		 * 	The new value for this property.
		 * 	<br>Cannot be <jk>null</jk>.
		 * @return This object.
		 */
		public Builder properties(Map<String,Object> value) {
			assertArgNotNull("value", value);
			properties.reset();
			properties.get().putAll(value);
			return this;
		}

		/**
		 * Adds a property to this session.
		 *
		 * @param key The property key.
		 * 	<br>Cannot be <jk>null</jk>.
		 * @param value The property value.
		 * 	<br>Can be <jk>null</jk> (removes the property).
		 * @return This object.
		 */
		public Builder property(String key, Object value) {
			assertArgNotNull("key", key);
			var map = properties.get();
			if (value == null) {
				map.remove(key);
			} else {
				map.put(key, value);
			}
			return this;
		}

		/**
		 * Create an unmodifiable session.
		 *
		 * <p>
		 * The created ContextSession object will be unmodifiable which makes it suitable for caching and reuse.
		 *
		 * @return This object.
		 */
		public Builder unmodifiable() {
			unmodifiable = true;
			return this;
		}
	}

	private final boolean debug;
	private final boolean unmodifiable;
	private final Context ctx;
	private final Map<String,Object> properties;
	private List<String> warnings;	// Any warnings encountered.

	/**
	 * Default constructor.
	 *
	 * @param builder The builder for this object.
	 * 	<br>Cannot be <jk>null</jk>.
	 */
	protected ContextSession(Builder builder) {
		assertArgNotNull("builder", builder);
		ctx = builder.ctx;
		debug = opt(builder.debug).orElse(ctx.isDebug());
		unmodifiable = builder.unmodifiable;
		var sp = builder.properties.get();
		if (unmodifiable) {
			properties = sp.isEmpty() ? Collections.emptyMap() : u(sp);
		} else {
			properties = sp;
		}
	}

	/**
	 * Logs a warning message.
	 *
	 * @param msg The warning message.
	 * 	<br>Cannot be <jk>null</jk>.
	 * @param args Optional {@link MessageFormat}-style arguments.
	 * 	<br>Cannot contain <jk>null</jk> values.
	 */
	public void addWarning(String msg, Object...args) {
		assertArgsNotNull("msg", msg, "args", args);
		if (unmodifiable)
			return;
		if (warnings == null)
			warnings = new LinkedList<>();
		warnings.add((warnings.size() + 1) + ": " + f(msg, args));
	}

	/**
	 * Throws a {@link BeanRuntimeException} if any warnings occurred in this session and debug is enabled.
	 */
	public void checkForWarnings() {
		if (debug && ! getWarnings().isEmpty())
			throw bex("Warnings occurred in session: \n" + join(getWarnings(), "\n"));
	}

	/**
	 * Returns the context that created this session.
	 *
	 * @return The context that created this session.
	 */
	public Context getContext() { return ctx; }

	/**
	 * Returns the session properties on this session.
	 *
	 * @return The session properties on this session.  Never <jk>null</jk>.
	 */
	public final Map<String,Object> getSessionProperties() { return properties; }

	/**
	 * Returns the warnings that occurred in this session.
	 *
	 * @return The warnings that occurred in this session, or <jk>null</jk> if no warnings occurred.
	 */
	public final List<String> getWarnings() { return warnings == null ? emptyList() : warnings; }

	/**
	 * Debug mode enabled.
	 *
	 * @see Context.Builder#debug()
	 * @return
	 * 	<jk>true</jk> if debug mode is enabled.
	 */
	public boolean isDebug() { return debug; }

	@Override /* Overridden from Object */
	public String toString() {
		return r(properties());
	}

	/**
	 * Returns the properties on this bean as a map for debugging.
	 *
	 * @return The properties on this bean as a map for debugging.
	 */
	protected FluentMap<String,Object> properties() {
		return filteredBeanPropertyMap()
			.a("debug", debug);
	}
}