/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.registry.aws;

import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import org.apache.nifi.registry.extension.BundleCoordinate;
import org.apache.nifi.registry.extension.BundlePersistenceContext;
import org.apache.nifi.registry.extension.BundlePersistenceException;
import org.apache.nifi.registry.extension.BundlePersistenceProvider;
import org.apache.nifi.registry.extension.BundleVersionCoordinate;
import org.apache.nifi.registry.extension.BundleVersionType;
import org.apache.nifi.registry.provider.ProviderConfigurationContext;
import org.apache.nifi.registry.provider.ProviderCreationException;
import org.apache.nifi.registry.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.model.DeleteObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.ListObjectsRequest;
import software.amazon.awssdk.services.s3.model.ListObjectsResponse;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.utils.IoUtils;
import software.amazon.awssdk.utils.StringUtils;

public class S3BundlePersistenceProvider
implements BundlePersistenceProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(S3BundlePersistenceProvider.class);
    public static final String REGION_PROP = "Region";
    public static final String BUCKET_NAME_PROP = "Bucket Name";
    public static final String KEY_PREFIX_PROP = "Key Prefix";
    public static final String CREDENTIALS_PROVIDER_PROP = "Credentials Provider";
    public static final String ACCESS_KEY_PROP = "Access Key";
    public static final String SECRET_ACCESS_KEY_PROP = "Secret Access Key";
    public static final String ENDPOINT_URL_PROP = "Endpoint URL";
    public static final String NAR_EXTENSION = ".nar";
    public static final String CPP_EXTENSION = ".cpp";
    private volatile S3Client s3Client;
    private volatile String s3BucketName;
    private volatile String s3KeyPrefix;

    public void onConfigured(ProviderConfigurationContext configurationContext) throws ProviderCreationException {
        this.s3BucketName = (String)configurationContext.getProperties().get(BUCKET_NAME_PROP);
        if (StringUtils.isBlank((CharSequence)this.s3BucketName)) {
            throw new ProviderCreationException("The property 'Bucket Name' must be provided");
        }
        String keyPrefixValue = (String)configurationContext.getProperties().get(KEY_PREFIX_PROP);
        this.s3KeyPrefix = StringUtils.isBlank((CharSequence)keyPrefixValue) ? null : keyPrefixValue;
        this.s3Client = this.createS3Client(configurationContext);
    }

    protected S3Client createS3Client(ProviderConfigurationContext configurationContext) {
        S3ClientBuilder builder = (S3ClientBuilder)((S3ClientBuilder)S3Client.builder().region(this.getRegion(configurationContext))).credentialsProvider(this.getCredentialsProvider(configurationContext));
        URI s3EndpointOverride = this.getS3EndpointOverride(configurationContext);
        if (s3EndpointOverride != null) {
            builder.endpointOverride(s3EndpointOverride);
        }
        return (S3Client)builder.build();
    }

    private Region getRegion(ProviderConfigurationContext configurationContext) {
        String regionValue = (String)configurationContext.getProperties().get(REGION_PROP);
        if (StringUtils.isBlank((CharSequence)regionValue)) {
            throw new ProviderCreationException("The property 'Region' must be provided");
        }
        Region region = null;
        for (Region r : Region.regions()) {
            if (!r.id().equals(regionValue)) continue;
            region = r;
            break;
        }
        if (region == null) {
            LOGGER.warn("The provided region was not found in the list of known regions. This may indicate an invalid region, or may indicate a region that is newer than the known list of regions");
            region = Region.of((String)regionValue);
        }
        LOGGER.debug("Using region {}", (Object)region.id());
        return region;
    }

    private AwsCredentialsProvider getCredentialsProvider(ProviderConfigurationContext configurationContext) {
        CredentialProvider credentialProvider;
        String credentialsProviderValue = (String)configurationContext.getProperties().get(CREDENTIALS_PROVIDER_PROP);
        if (StringUtils.isBlank((CharSequence)credentialsProviderValue)) {
            throw new ProviderCreationException("The property 'Credentials Provider' must be provided");
        }
        try {
            credentialProvider = CredentialProvider.valueOf(credentialsProviderValue);
        }
        catch (Exception e) {
            throw new ProviderCreationException("The property 'Credentials Provider' must be one of [" + String.valueOf((Object)CredentialProvider.STATIC) + ", " + String.valueOf((Object)CredentialProvider.DEFAULT_CHAIN) + " ]");
        }
        if (CredentialProvider.STATIC == credentialProvider) {
            String accesKeyValue = (String)configurationContext.getProperties().get(ACCESS_KEY_PROP);
            String secretAccessKey = (String)configurationContext.getProperties().get(SECRET_ACCESS_KEY_PROP);
            if (StringUtils.isBlank((CharSequence)accesKeyValue) || StringUtils.isBlank((CharSequence)secretAccessKey)) {
                throw new ProviderCreationException("The properties 'Access Key' and 'Secret Access Key' must be provided when using " + String.valueOf((Object)CredentialProvider.STATIC) + " credentials provider");
            }
            LOGGER.debug("Creating StaticCredentialsProvider");
            AwsBasicCredentials awsCredentials = AwsBasicCredentials.create((String)accesKeyValue, (String)secretAccessKey);
            return StaticCredentialsProvider.create((AwsCredentials)awsCredentials);
        }
        LOGGER.debug("Creating DefaultCredentialsProvider");
        return DefaultCredentialsProvider.builder().build();
    }

    private URI getS3EndpointOverride(ProviderConfigurationContext configurationContext) {
        URI s3EndpointOverride;
        String endpointUrlValue = (String)configurationContext.getProperties().get(ENDPOINT_URL_PROP);
        try {
            s3EndpointOverride = StringUtils.isBlank((CharSequence)endpointUrlValue) ? null : URI.create(endpointUrlValue);
        }
        catch (IllegalArgumentException e) {
            String errMessage = "The optional property 'Endpoint URL' must be a valid URL if set. URI Syntax Exception is: " + e.getLocalizedMessage();
            LOGGER.error(errMessage);
            LOGGER.debug("", (Throwable)e);
            throw new ProviderCreationException(errMessage, (Throwable)e);
        }
        return s3EndpointOverride;
    }

    public synchronized void createBundleVersion(BundlePersistenceContext context, InputStream contentStream) throws BundlePersistenceException {
        this.createOrUpdateBundleVersion(context, contentStream);
    }

    public synchronized void updateBundleVersion(BundlePersistenceContext context, InputStream contentStream) throws BundlePersistenceException {
        this.createOrUpdateBundleVersion(context, contentStream);
    }

    private synchronized void createOrUpdateBundleVersion(BundlePersistenceContext context, InputStream contentStream) throws BundlePersistenceException {
        String key = this.getKey(context.getCoordinate());
        LOGGER.debug("Saving bundle version to S3 in bucket '{}' with key '{}'", (Object)this.s3BucketName, (Object)key);
        PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(this.s3BucketName).key(key).build();
        RequestBody requestBody = RequestBody.fromInputStream((InputStream)contentStream, (long)context.getSize());
        try {
            this.s3Client.putObject(request, requestBody);
            LOGGER.debug("Successfully saved bundle version to S3 bucket '{}' with key '{}'", (Object)this.s3BucketName, (Object)key);
        }
        catch (Exception e) {
            throw new BundlePersistenceException("Error saving bundle version to S3 due to: " + e.getMessage(), (Throwable)e);
        }
    }

    public synchronized void getBundleVersionContent(BundleVersionCoordinate versionCoordinate, OutputStream outputStream) throws BundlePersistenceException {
        String key = this.getKey(versionCoordinate);
        LOGGER.debug("Retrieving bundle version from S3 bucket '{}' with key '{}'", (Object)this.s3BucketName, (Object)key);
        GetObjectRequest request = (GetObjectRequest)GetObjectRequest.builder().bucket(this.s3BucketName).key(key).build();
        try (ResponseInputStream response = this.s3Client.getObject(request);){
            IoUtils.copy((InputStream)response, (OutputStream)outputStream);
            LOGGER.debug("Successfully retrieved bundle version from S3 bucket '{}' with key '{}'", (Object)this.s3BucketName, (Object)key);
        }
        catch (Exception e) {
            throw new BundlePersistenceException("Error retrieving bundle version from S3 due to: " + e.getMessage(), (Throwable)e);
        }
    }

    public synchronized void deleteBundleVersion(BundleVersionCoordinate versionCoordinate) throws BundlePersistenceException {
        String key = this.getKey(versionCoordinate);
        LOGGER.debug("Deleting bundle version from S3 bucket '{}' with key '{}'", (Object)this.s3BucketName, (Object)key);
        DeleteObjectRequest request = (DeleteObjectRequest)DeleteObjectRequest.builder().bucket(this.s3BucketName).key(key).build();
        try {
            this.s3Client.deleteObject(request);
            LOGGER.debug("Successfully deleted bundle version from S3 bucket '{}' with key '{}'", (Object)this.s3BucketName, (Object)key);
        }
        catch (Exception e) {
            throw new BundlePersistenceException("Error deleting bundle version from S3 due to: " + e.getMessage(), (Throwable)e);
        }
    }

    public synchronized void deleteAllBundleVersions(BundleCoordinate bundleCoordinate) throws BundlePersistenceException {
        String basePrefix = this.s3KeyPrefix == null ? "" : this.s3KeyPrefix + "/";
        String bundlePrefix = this.getBundlePrefix(bundleCoordinate.getBucketId(), bundleCoordinate.getGroupId(), bundleCoordinate.getArtifactId());
        String prefix = basePrefix + bundlePrefix;
        LOGGER.debug("Deleting all bundle versions from S3 bucket '{}' with prefix '{}'", (Object)this.s3BucketName, (Object)prefix);
        try {
            ListObjectsResponse objectsResponse = this.s3Client.listObjects((ListObjectsRequest)ListObjectsRequest.builder().bucket(this.s3BucketName).prefix(prefix).build());
            for (S3Object s3Object : objectsResponse.contents()) {
                String s3ObjectKey = s3Object.key();
                this.s3Client.deleteObject((DeleteObjectRequest)DeleteObjectRequest.builder().bucket(this.s3BucketName).key(s3ObjectKey).build());
                LOGGER.debug("Successfully object from S3 bucket '{}' with key '{}'", (Object)this.s3BucketName, (Object)s3ObjectKey);
            }
            LOGGER.debug("Successfully deleted all bundle versions from S3 bucket '{}' with prefix '{}'", (Object)this.s3BucketName, (Object)prefix);
        }
        catch (Exception e) {
            throw new BundlePersistenceException("Error deleting bundle versions from S3 due to: " + e.getMessage(), (Throwable)e);
        }
    }

    public void preDestruction() {
        this.s3Client.close();
    }

    private String getKey(BundleVersionCoordinate coordinate) {
        String bundlePrefix = this.getBundlePrefix(coordinate.getBucketId(), coordinate.getGroupId(), coordinate.getArtifactId());
        String sanitizedArtifact = S3BundlePersistenceProvider.sanitize(coordinate.getArtifactId());
        String sanitizedVersion = S3BundlePersistenceProvider.sanitize(coordinate.getVersion());
        String bundleFileExtension = S3BundlePersistenceProvider.getBundleFileExtension(coordinate.getType());
        String bundleFilename = sanitizedArtifact + "-" + sanitizedVersion + bundleFileExtension;
        String key = bundlePrefix + "/" + sanitizedVersion + "/" + bundleFilename;
        if (this.s3KeyPrefix == null) {
            return key;
        }
        return this.s3KeyPrefix + "/" + key;
    }

    private String getBundlePrefix(String bucketId, String groupId, String artifactId) {
        String sanitizedBucketId = S3BundlePersistenceProvider.sanitize(bucketId);
        String sanitizedGroup = S3BundlePersistenceProvider.sanitize(groupId);
        String sanitizedArtifact = S3BundlePersistenceProvider.sanitize(artifactId);
        return sanitizedBucketId + "/" + sanitizedGroup + "/" + sanitizedArtifact;
    }

    private static String sanitize(String input) {
        return FileUtils.sanitizeFilename((String)input).trim().toLowerCase();
    }

    static String getBundleFileExtension(BundleVersionType bundleType) {
        return switch (bundleType) {
            default -> throw new MatchException(null, null);
            case BundleVersionType.NIFI_NAR -> NAR_EXTENSION;
            case BundleVersionType.MINIFI_CPP -> CPP_EXTENSION;
        };
    }

    public static enum CredentialProvider {
        STATIC,
        DEFAULT_CHAIN;

    }
}

