"use strict";
/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SetSerializer = exports.ListSerializer = exports.MapSerializer = exports.BulkSetSerializer = exports.DirectionSerializer = exports.TSerializer = exports.Path3Serializer = exports.PathSerializer = exports.EdgeSerializer = exports.PropertySerializer = exports.VertexPropertySerializer = exports.VertexSerializer = exports.TraversalStrategySerializer = exports.TraverserSerializer = exports.EnumSerializer = exports.LambdaSerializer = exports.TextPSerializer = exports.PSerializer = exports.BytecodeSerializer = exports.LongSerializer = exports.DateSerializer = exports.NumberSerializer = exports.TypeSerializer = exports.typeKey = exports.valueKey = void 0;
/**
 * @author Jorge Bay Gondra
 */
const t = __importStar(require("../../process/traversal.cjs"));
const ts = __importStar(require("../../process/traversal-strategy.cjs"));
const bytecode_js_1 = __importDefault(require("../../process/bytecode.cjs"));
const g = __importStar(require("../graph.cjs"));
const utils = __importStar(require("../../utils.cjs"));
exports.valueKey = '@value';
exports.typeKey = '@type';
/**
 * @abstract
 */
class TypeSerializer {
    reader;
    writer;
    serialize(value) {
        throw new Error('serialize() method not implemented for ' + this.constructor.name);
    }
    deserialize(value) {
        throw new Error('deserialize() method not implemented for ' + this.constructor.name);
    }
    canBeUsedFor(value) {
        throw new Error('canBeUsedFor() method not implemented for ' + this.constructor.name);
    }
}
exports.TypeSerializer = TypeSerializer;
class NumberSerializer extends TypeSerializer {
    serialize(item) {
        if (isNaN(item)) {
            return {
                [exports.typeKey]: 'g:Double',
                [exports.valueKey]: 'NaN',
            };
        }
        else if (item === Number.POSITIVE_INFINITY) {
            return {
                [exports.typeKey]: 'g:Double',
                [exports.valueKey]: 'Infinity',
            };
        }
        else if (item === Number.NEGATIVE_INFINITY) {
            return {
                [exports.typeKey]: 'g:Double',
                [exports.valueKey]: '-Infinity',
            };
        }
        return item;
    }
    deserialize(obj) {
        const val = obj[exports.valueKey];
        if (val === 'NaN') {
            return NaN;
        }
        else if (val === 'Infinity') {
            return Number.POSITIVE_INFINITY;
        }
        else if (val === '-Infinity') {
            return Number.NEGATIVE_INFINITY;
        }
        return parseFloat(val);
    }
    canBeUsedFor(value) {
        return typeof value === 'number';
    }
}
exports.NumberSerializer = NumberSerializer;
class DateSerializer extends TypeSerializer {
    serialize(item) {
        return {
            [exports.typeKey]: 'g:Date',
            [exports.valueKey]: item.getTime(),
        };
    }
    deserialize(obj) {
        return new Date(obj[exports.valueKey]);
    }
    canBeUsedFor(value) {
        return value instanceof Date;
    }
}
exports.DateSerializer = DateSerializer;
class LongSerializer extends TypeSerializer {
    serialize(item) {
        return {
            [exports.typeKey]: 'g:Int64',
            [exports.valueKey]: item.value,
        };
    }
    canBeUsedFor(value) {
        return value instanceof utils.Long;
    }
}
exports.LongSerializer = LongSerializer;
class BytecodeSerializer extends TypeSerializer {
    serialize(item) {
        let bytecode = item;
        if (item instanceof t.Traversal) {
            bytecode = item.getBytecode();
        }
        const result = {};
        result[exports.typeKey] = 'g:Bytecode';
        const resultValue = (result[exports.valueKey] = {});
        const sources = this._serializeInstructions(bytecode.sourceInstructions);
        if (sources) {
            resultValue['source'] = sources;
        }
        const steps = this._serializeInstructions(bytecode.stepInstructions);
        if (steps) {
            resultValue['step'] = steps;
        }
        return result;
    }
    _serializeInstructions(instructions) {
        if (instructions.length === 0) {
            return null;
        }
        const result = new Array(instructions.length);
        result[0] = instructions[0];
        for (let i = 0; i < instructions.length; i++) {
            result[i] = instructions[i].map((item) => this.writer.adaptObject(item));
        }
        return result;
    }
    canBeUsedFor(value) {
        return value instanceof bytecode_js_1.default || value instanceof t.Traversal;
    }
}
exports.BytecodeSerializer = BytecodeSerializer;
class PSerializer extends TypeSerializer {
    /** @param {P} item */
    serialize(item) {
        const result = {};
        result[exports.typeKey] = 'g:P';
        const resultValue = (result[exports.valueKey] = {
            predicate: item.operator,
        });
        if (item.other === undefined || item.other === null) {
            resultValue['value'] = this.writer.adaptObject(item.value);
        }
        else {
            resultValue['value'] = [this.writer.adaptObject(item.value), this.writer.adaptObject(item.other)];
        }
        return result;
    }
    canBeUsedFor(value) {
        return value instanceof t.P;
    }
}
exports.PSerializer = PSerializer;
class TextPSerializer extends TypeSerializer {
    /** @param {TextP} item */
    serialize(item) {
        const result = {};
        result[exports.typeKey] = 'g:TextP';
        const resultValue = (result[exports.valueKey] = {
            predicate: item.operator,
        });
        if (item.other === undefined || item.other === null) {
            resultValue['value'] = this.writer.adaptObject(item.value);
        }
        else {
            resultValue['value'] = [this.writer.adaptObject(item.value), this.writer.adaptObject(item.other)];
        }
        return result;
    }
    canBeUsedFor(value) {
        return value instanceof t.TextP;
    }
}
exports.TextPSerializer = TextPSerializer;
class LambdaSerializer extends TypeSerializer {
    /** @param {Function} item */
    serialize(item) {
        const lambdaDef = item();
        // check if the language is specified otherwise assume gremlin-groovy.
        const returnIsString = typeof lambdaDef === 'string';
        const script = returnIsString ? lambdaDef : lambdaDef[0];
        const lang = returnIsString ? 'gremlin-groovy' : lambdaDef[1];
        // detect argument count
        const argCount = lang === 'gremlin-groovy' && script.includes('->')
            ? script.substring(0, script.indexOf('->')).includes(',')
                ? 2
                : 1
            : -1;
        return {
            [exports.typeKey]: 'g:Lambda',
            [exports.valueKey]: {
                arguments: argCount,
                language: lang,
                script: script,
            },
        };
    }
    canBeUsedFor(value) {
        return typeof value === 'function';
    }
}
exports.LambdaSerializer = LambdaSerializer;
class EnumSerializer extends TypeSerializer {
    /** @param {EnumValue} item */
    serialize(item) {
        return {
            [exports.typeKey]: 'g:' + item.typeName,
            [exports.valueKey]: item.elementName,
        };
    }
    canBeUsedFor(value) {
        return value && value.typeName && value instanceof t.EnumValue;
    }
}
exports.EnumSerializer = EnumSerializer;
class TraverserSerializer extends TypeSerializer {
    /** @param {Traverser} item */
    serialize(item) {
        return {
            [exports.typeKey]: 'g:Traverser',
            [exports.valueKey]: {
                value: this.writer.adaptObject(item.object),
                bulk: this.writer.adaptObject(item.bulk),
            },
        };
    }
    deserialize(obj) {
        const value = obj[exports.valueKey];
        return new t.Traverser(this.reader.read(value['value']), this.reader.read(value['bulk']));
    }
    canBeUsedFor(value) {
        return value instanceof t.Traverser;
    }
}
exports.TraverserSerializer = TraverserSerializer;
class TraversalStrategySerializer extends TypeSerializer {
    /** @param {TraversalStrategy} item */
    serialize(item) {
        const conf = {};
        for (const k in item.configuration) {
            if (item.configuration.hasOwnProperty(k)) {
                conf[k] = this.writer.adaptObject(item.configuration[k]);
            }
        }
        return {
            [exports.typeKey]: 'g:' + item.constructor.name,
            [exports.valueKey]: conf,
        };
    }
    canBeUsedFor(value) {
        return value instanceof ts.TraversalStrategy;
    }
}
exports.TraversalStrategySerializer = TraversalStrategySerializer;
class VertexSerializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        return new g.Vertex(this.reader.read(value['id']), value['label'], this.reader.read(value['properties']));
    }
    /** @param {Vertex} item */
    serialize(item) {
        return {
            [exports.typeKey]: 'g:Vertex',
            [exports.valueKey]: {
                id: this.writer.adaptObject(item.id),
                label: item.label,
            },
        };
    }
    canBeUsedFor(value) {
        return value instanceof g.Vertex;
    }
}
exports.VertexSerializer = VertexSerializer;
class VertexPropertySerializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        return new g.VertexProperty(this.reader.read(value['id']), value['label'], this.reader.read(value['value']), this.reader.read(value['properties']));
    }
}
exports.VertexPropertySerializer = VertexPropertySerializer;
class PropertySerializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        return new g.Property(value['key'], this.reader.read(value['value']));
    }
}
exports.PropertySerializer = PropertySerializer;
class EdgeSerializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        return new g.Edge(this.reader.read(value['id']), new g.Vertex(this.reader.read(value['outV']), this.reader.read(value['outVLabel'])), value['label'], new g.Vertex(this.reader.read(value['inV']), this.reader.read(value['inVLabel'])), this.reader.read(value['properties']));
    }
    /** @param {Edge} item */
    serialize(item) {
        return {
            [exports.typeKey]: 'g:Edge',
            [exports.valueKey]: {
                id: this.writer.adaptObject(item.id),
                label: item.label,
                outV: this.writer.adaptObject(item.outV.id),
                outVLabel: item.outV.label,
                inV: this.writer.adaptObject(item.inV.id),
                inVLabel: item.inV.label,
            },
        };
    }
    canBeUsedFor(value) {
        return value instanceof g.Edge;
    }
}
exports.EdgeSerializer = EdgeSerializer;
class PathSerializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        const objects = value['objects'].map((o) => this.reader.read(o));
        return new g.Path(this.reader.read(value['labels']), objects);
    }
}
exports.PathSerializer = PathSerializer;
class Path3Serializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        return new g.Path(this.reader.read(value['labels']), this.reader.read(value['objects']));
    }
}
exports.Path3Serializer = Path3Serializer;
class TSerializer extends TypeSerializer {
    deserialize(obj) {
        return t.t[obj[exports.valueKey]];
    }
}
exports.TSerializer = TSerializer;
class DirectionSerializer extends TypeSerializer {
    deserialize(obj) {
        return t.direction[obj[exports.valueKey].toLowerCase()];
    }
}
exports.DirectionSerializer = DirectionSerializer;
class ArraySerializer extends TypeSerializer {
    typeKey;
    constructor(typeKey) {
        super();
        this.typeKey = typeKey;
    }
    deserialize(obj) {
        const value = obj[exports.valueKey];
        if (!Array.isArray(value)) {
            throw new Error('Expected Array, obtained: ' + value);
        }
        return value.map((x) => this.reader.read(x));
    }
    /** @param {Array} item */
    serialize(item) {
        return {
            [exports.typeKey]: this.typeKey,
            [exports.valueKey]: item.map((x) => this.writer.adaptObject(x)),
        };
    }
    canBeUsedFor(value) {
        return Array.isArray(value);
    }
}
class BulkSetSerializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        if (!Array.isArray(value)) {
            throw new Error('Expected Array, obtained: ' + value);
        }
        // coerce the BulkSet to List. if the bulk exceeds the int space then we can't coerce to List anyway,
        // so this query will be trouble. we'd need a legit BulkSet implementation here in js. this current
        // implementation is here to replicate the previous functionality that existed on the server side in
        // previous versions.
        let result = [];
        for (let ix = 0, iy = value.length; ix < iy; ix += 2) {
            const pair = value.slice(ix, ix + 2);
            result = result.concat(Array(this.reader.read(pair[1])).fill(this.reader.read(pair[0])));
        }
        return result;
    }
}
exports.BulkSetSerializer = BulkSetSerializer;
class MapSerializer extends TypeSerializer {
    deserialize(obj) {
        const value = obj[exports.valueKey];
        if (!Array.isArray(value)) {
            throw new Error('Expected Array, obtained: ' + value);
        }
        const result = new Map();
        for (let i = 0; i < value.length; i += 2) {
            result.set(this.reader.read(value[i]), this.reader.read(value[i + 1]));
        }
        return result;
    }
    /** @param {Map} map */
    serialize(map) {
        const arr = [];
        map.forEach((v, k) => {
            arr.push(this.writer.adaptObject(k));
            arr.push(this.writer.adaptObject(v));
        });
        return {
            [exports.typeKey]: 'g:Map',
            [exports.valueKey]: arr,
        };
    }
    canBeUsedFor(value) {
        return value instanceof Map;
    }
}
exports.MapSerializer = MapSerializer;
class ListSerializer extends ArraySerializer {
    constructor() {
        super('g:List');
    }
}
exports.ListSerializer = ListSerializer;
class SetSerializer extends ArraySerializer {
    constructor() {
        super('g:Set');
    }
    deserialize(obj) {
        return new Set(super.deserialize(obj));
    }
    serialize(item) {
        return {
            [exports.typeKey]: this.typeKey,
            [exports.valueKey]: [...item].map((x) => this.writer.adaptObject(x)),
        };
    }
    canBeUsedFor(value) {
        return value instanceof Set;
    }
}
exports.SetSerializer = SetSerializer;
